# Packages ----------------------------------------------------------------

dir <- "Insert directory here"

packagesNeeded <- c("tidyverse",
                    "data.table",
                    "mpi", 
                    "lubridate", 
                    "bankRun",
                    "ggrba",
                    "arphit",
                    "rbatools",
                    "mpi", 
                    "tseries", 
                    "janitor", 
                    "lmtest", 
                    "sandwich",
                    "minpack.lm", 
                    "msm", 
                    "ivreg", 
                    "fixest", 
                    "ggtext")

if(any(!packagesNeeded %in% dplyr::pull(tibble::as_tibble(installed.packages()), Package))){
  
  install.packages(packagesNeeded[!packagesNeeded %in% dplyr::pull(tibble::as_tibble(installed.packages()), Package)])
  
}

for(i in packagesNeeded){library(i, character.only = TRUE)}


# Data --------------------------------------------------------------------

finalData <- read.csv(paste0(dir, "Data for Estimation.xlsx")) %>% 
  as_tibble() %>% 
  select(-X)

# Main regressions -------------------------------------------------------------

## IV -----------------------------------------------------------

IVData <- finalData  %>% 
  filter(tradeDate >= as.Date("2020-11-05"), # a couple days after BPP commenced (3 Nov 2020)
         tradeDate <= as.Date("2024-08-31")) %>%  
  mutate(depositMeasure = TotalDepositsAPRA)

IVRegForPred <- fixest::feols(data = IVData, 
                              fml = APRASOFIASpread ~ 
                                log(depositMeasure) +
                                APRAVolumeGCMinus10 
                              | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                              vcov = sandwich::vcovHAC)

IVRegForPredJustESBal <- fixest::feols(data = IVData, 
                                       fml = APRASOFIASpread ~ 
                                         1 | log(ESBal) ~ log(ESBal - OMO), 
                                       vcov = sandwich::vcovHAC)

## Predictions ------------------------------------------------------------

spreadToESPredictedIV <- tibble(tradeDate = IVData$tradeDate[!is.na(IVData$SOFIAspread)], 
                                IVPredJustESBal = IVRegForPredJustESBal$fitted.values, 
                                IVPredCD = IVRegForPred$fitted.values)

aIVCD <- IVRegForPred$coefficients[1]
bIVCD <- IVRegForPred$coefficients[2]
cIVCD <- IVRegForPred$coefficients[3]
dIVCD <- IVRegForPred$coefficients[4]

aIV <- IVRegForPredJustESBal$coefficients[1]
bIV <- IVRegForPredJustESBal$coefficients[2]

# Robustness --------------------------------------------------------------

robustnessData <- finalData  %>% 
  left_join(uninsuredDeposits, by = c("tradeDate" = "asAt")) %>% 
  filter(tradeDate >= as.Date("2020-11-05")) %>%  # a couple days after BPP commenced (3 Nov 2020)
  mutate(depositMeasure = TotalDepositsAPRA)

## Base reg ----------------------------------------------------------------

IVRegBase <- fixest::feols(data = robustnessData, 
                           fml = APRASOFIASpread ~ csw0(log(depositMeasure), 
                                                        APRAVolumeGCMinus10)
                           | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                           vcov = sandwich::vcovHAC)

OLSRegBase <- fixest::feols(data = robustnessData, 
                            fml = APRASOFIASpread ~ csw0(log(depositMeasure), 
                                                         APRAVolumeGCMinus10) + 
                              log(ESBal), 
                            vcov = sandwich::vcovHAC)

## Deposit measures --------------------------------------------------------

robustnessDep <- fixest::feols(data = robustnessData, 
                               fml = APRASOFIASpread ~ 
                                 sw(log(TotalDepositsAPRA), 
                                    log(atCall), 
                                    log(overnightFunding), 
                                    log(UninsuredTotal)) + 
                                 APRAVolumeGCMinus10
                               | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                               vcov = sandwich::vcovHAC)

## Collateral dynamics measures --------------------------------------------

robustnessCD <- fixest::feols(data = robustnessData, 
                              fml = APRASOFIASpread ~ 
                                log(TotalDepositsAPRA) + 
                                sw(APRAVolumeGCMinus10, 
                                   bondOIS10year, 
                                   bondOIS3year, 
                                   ASW)
                              | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                              vcov = sandwich::vcovHAC)

specialsThresholdAnalysisList <- list()

for(i in c(5, 10, 15, 20)){
  
  specialsThresholdAnalysisList[[i]] <- repoRatesByEntityCurrent %>%
    mutate(marketType = "Private") %>% 
    bind_rows(RBASLToBind) %>% 
    dplyr::group_by(.data$tradeDate) %>%
    mutate(shareVol = collMV/sum(collMV)*100) %>% 
    dplyr::arrange(tradeDate, rate, shareVol) %>% 
    mutate(cumulativeQ = cumsum(shareVol)) %>%
    left_join(APRASOFIACalc, by = "tradeDate") %>% 
    mutate(specialsSpread = (APRASOFIA - rate)*100) %>% 
    select(tradeDate, rate, APRASOFIA, specialsSpread, collMV, everything()) %>% 
    filter(specialsSpread > i) %>% 
    group_by(tradeDate) %>% 
    summarise(APRARepoVolume = max(cumulativeQ)) %>% 
    ungroup() %>% 
    mutate(cutoff = i)
  
}

specialsThresholdAnalysis <- bind_rows(specialsThresholdAnalysisList) %>% 
  pivot_wider(names_from = c("cutoff"), 
              names_sep = "_", 
              values_from = APRARepoVolume) %>% 
  `colnames<-`(c("tradeDate", paste0("Specials_", colnames(.)[!grepl("tradeDate", colnames(.))])))

robustnessSpecials <- fixest::feols(data = robustnessData %>% 
                                      left_join(specialsThresholdAnalysis), 
                                    fml = APRASOFIASpread ~ 
                                      log(TotalDepositsAPRA) + 
                                      sw(Specials_5, 
                                         Specials_10, 
                                         Specials_15, 
                                         Specials_20)
                                    | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                                    vcov = sandwich::vcovHAC)

## Adding credit lines ------------------------------------------

robustnessCreditLines <- fixest::feols(data = robustnessData %>% 
                                         filter(!is.na(creditLines)), 
                                       fml = APRASOFIASpread ~ 
                                         log(TotalDepositsAPRA) +
                                         sw0(log(creditLines)) + 
                                         APRAVolumeGCMinus10 
                                       | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                                       vcov = sandwich::vcovHAC)

## Surplus vs. total -------------------------------------------------------

robustnessTotalESBal <- fixest::feols(data = robustnessData, 
                                      fml = APRASOFIASpread ~ 
                                        TotalDepositsAPRA + 
                                        APRAVolumeGCMinus10
                                      | log(ESBalTotal) ~ log(ESBalTotal - OMO - foreignDep), 
                                      vcov = sandwich::vcovHAC)

## Rel value ---------------------------------------------------------------

robustnessBOB <- fixest::feols(data = robustnessData %>% 
                                 mutate(HQLADyn = ASW + BOB), 
                               fml = APRASOFIASpread ~ 
                                 log(TotalDepositsAPRA) + 
                                 APRAVolumeGCMinus10 +
                                 sw(ASW, BOB, HQLADyn) | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                               vcov = sandwich::vcovHAC)

cor(robustnessData$ASW, 
    robustnessData$APRAVolumeGCMinus10)


robustnessBOBASW <- fixest::feols(data = robustnessData %>%
                                    mutate(HQLADyn = ASW + BOB),
                                  fml = APRASOFIASpread ~
                                    log(TotalDepositsAPRA) + 
                                    ASW | log(ESBal) ~ log(ESBal - OMO - foreignDep),
                                  vcov = sandwich::vcovHAC)

## ASX vs. APRA ------------------------------------------------------------

robustnessSOFIA <- fixest::feols(data = robustnessData , 
                                 fml = c(SOFIAspreadMedian, APRASOFIASpread) ~ 
                                   log(TotalDepositsAPRA) +
                                   APRAVolumeGCMinus10
                                 | log(ESBal) ~ log(ESBal - OMO - foreignDep), 
                                 vcov = sandwich::vcovHAC)

## Functional forms --------------------------------------------------------

robustnessFFInverse <- fixest::feols(data = robustnessData, 
                                     fml = APRASOFIASpread ~ log(depositMeasure) + 
                                       APRAVolumeGCMinus10 + 
                                       I(1/ESBal), 
                                     vcov = "HC1")

robustnessFFArcTan <- fixest::feols(data = robustnessData, 
                                    fml = APRASOFIASpread ~ 
                                      atan(ESBal) +
                                      log(TotalDepositsAPRA) + 
                                      APRAVolumeGCMinus10, 
                                    vcov = "HC1"
)

## Export ---------------------------------------------------

robTables <- list(IVRegBase, 
                  OLSRegBase, 
                  robustnessDep, 
                  robustnessCD, 
                  robustnessCreditLines, 
                  robustnessTotalESBal, 
                  robustnessSpecials,
                  robustnessBOB, 
                  robustnessSOFIA,
                  robustnessFFInverse, 
                  robustnessFFArcTan)

names(robTables) <- c("Base", 
                      "OLS",
                      "Deposits", 
                      "CD", 
                      "Credit Lines", 
                      "Total ES Bal", 
                      "Specials", 
                      "Relative val", 
                      "SOFIA", 
                      "Inverse ff", 
                      "Arctan ff")

robOutputs <- list()

for(i in 1:length(robTables)){
  
  robOutputs[[i]] <- fixest::etable(robTables[[i]], 
                                    tex = FALSE, 
                                    digits = 4, 
                                    digits.stats = 4, 
                                    fitstat = c("n", "aic", "ar2", "ll"),
                                    signif.code = c("***" = 0.01, 
                                                    "**" = 0.05, 
                                                    "*" = 0.10))
  
}

wb <- openxlsx::createWorkbook()

sheetnames <- names(robTables)

lapply(seq(from = 1, by = 1, to = length(sheetnames)), function(x) {
  
  openxlsx::addWorksheet(wb, sheetnames[x])
  openxlsx::writeData(wb, sheet = sheetnames[x], robOutputs[[x]])
  
})

openxlsx::saveWorkbook(wb, file = paste0(dir, "Regression Outputs.xlsx"), overwrite = TRUE)

# Graphs  -----------------------------------------------------------------

## Graph 1 - Overnight repo rate and ES balances ---------------------------

graphNo <- 1

ONRepoAndESBalPanel1 <- finalData %>% 
  select(tradeDate, APRASOFIASpread, cashRateSpread) %>% 
  
  mutate(cashRateSpread = cashRateSpread*100) %>% 
  `colnames<-`(c("tradeDate", "Overnight repo rate", "Cash rate")) %>% 
  pivot_longer(-tradeDate) %>% 
  ggrba(aes(x = tradeDate, y = value, colour = name, .autolabel = name)) + 
  geom_line() +
  scale_color_manual(values = as.vector(c(RBA["Red1"], 
                                          RBA["Blue5"]))) +
  labs(title = "Spread to ES rate") + 
  scale_y_continuous_rba(units = "bps", limits = c(-10, 50)) 

ONMMAndESBalPanel2 <- finalData %>% 
  select(tradeDate, ESBal) %>% 
  ggrba() + 
  geom_line(aes(x = tradeDate, y = ESBal, colour = "Excess reserves")) +
  labs(title = "Excess reserves") +
  scale_color_manual(values = as.vector(c(RBA["Blue7"]))) + 
  scale_y_continuous_rba(units = "$b", limits = c(0, 500)) 

ONMMAndESBalGraph <- ONRepoAndESBalPanel1 / 
  ONMMAndESBalPanel2 + 
  multipanel_title_rba(title = "Money Market Spreads and the Level of Reserves",
                       subtitle = "September 2019 - Present") +
  source_rba(c("RBA")) +
  autolabel_var_rba()

ONMMAndESBalGraphLabelled <- ggrba::autolabel_rba(ONMMAndESBalGraph)

ONMMAndESBalGraphRDP <- format_rdp(ONMMAndESBalGraphLabelled)

ggrba::ggsave_rba(ONMMAndESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - Money Market Spreads and the Level of Reserves.png"))
ggrba::ggsave_rba(ONMMAndESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - Money Market Spreads and the Level of Reserves.svg"))
ggrba::ggsave_rba(ONMMAndESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - Money Market Spreads and the Level of Reserves.pdf"))

ggrba::export_excel_rba(gg = ONMMAndESBalGraphRDP, 
                        filename = paste0(dir, "Figure 1.xlsx"))

## Graph 2 - Time-varying reserve demand -------------------------------------

graphNo <- graphNo + 1

reserveDemandInstability <- finalData %>%
  mutate(group = case_when(tradeDate < as.Date("2022-01-31") ~ paste0("Sep 2019", "\U2012", "Dec 2021"), 
                           TRUE ~ paste0("Jan 2022", "\U2012", "Aug 2024"))) %>% 
  ggrba(aes(x = ESBal, y = APRASOFIASpread, label = tradeDate, colour = group, 
            .autolabel = group)) + 
  geom_point() + 
  geom_segment(aes(
    xend = c(tail(ESBal, n = -1), NA), 
    yend = c(tail(APRASOFIASpread, n = -1), NA)
  )) +
  labs(
    subtitle = "Monthly; September 2019-August 2024",
    title = "Instability in the Relationship between Money Market Spreads and Reserves",
    x = "Excess reserves \U2012 $b", 
    y = "Overnight repo rate spread to ES rate \U2012 bps") + 
  scale_y_continuous_rba(units = "50", limits = c(-10, 50), 
                         sec.axis = rhs_axis_rba("", scalar = 1, 
                                                 labels = rep("", 7))) + 
  scale_x_continuous_rba(units = "500") + 
  ggrba::axis_titles_rba() +
  source_rba(c("APRA", "RBA")) + 
  scale_color_manual(values = as.vector(c(RBA["Blue1"], 
                                          RBA["Brown4"]))) 

reserveDemandInstabilityLabelled <- ggrba::autolabel_rba(reserveDemandInstability)

reserveDemandInstabilityRDP <- format_rdp(reserveDemandInstabilityLabelled)

ggrba::ggsave_rba(reserveDemandInstabilityRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Time-varying relationship between the overnight repo rate and reserves", ".png"))
ggrba::ggsave_rba(reserveDemandInstabilityRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Time-varying relationship between the overnight repo rate and reserves", ".svg"))
ggrba::ggsave_rba(reserveDemandInstabilityRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Time-varying relationship between the overnight repo rate and reserves", ".pdf"))

ggrba::export_excel_rba(gg = reserveDemandInstabilityRDP, 
                        filename = paste0(dir, "Figure 2.xlsx"))

## Graph 3 - Deposits -----------------------------------------------------

graphNo <- graphNo + 1

depositsGraph <- depositsLongTimeSeries %>% 
  filter(period >= as.Date("2017-01-01")) %>% 
  ggrba() + 
  labs(
    title = "Total Bank Deposits", 
    subtitle = "Institutions with greater than $4 billion in deposits"
  ) +
  geom_line(aes(x = period, y = log(TotalDepositsAPRA)), 
            colour = RBA["DarkGreen1"]) +
  scale_y_continuous_rba(units = "log") + 
  source_rba(c("APRA"))

depositsGraphRDP <- format_rdp(depositsGraph)

ggrba::ggsave_rba(depositsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Total Bank Deposits", ".png"))
ggrba::ggsave_rba(depositsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Total Bank Deposits", ".svg"))
ggrba::ggsave_rba(depositsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Total Bank Deposits", ".pdf"))

ggrba::export_excel_rba(gg = depositsGraphRDP, 
                        filename = paste0(dir, "Figure 3.xlsx"))

## Graph 4 - Payments ------------------------------------------------------

graphNo <- graphNo + 1

paymentsGraph <- payments %>% 
  select(date, value) %>% 
  filter(date >= as.Date("2015-01-01")) %>% 
  ggrba() + 
  labs(
    title = "Total Payments", 
    subtitle = "blah"
  ) +
  geom_line(aes(x = date, y = value/1e3), 
            colour = RBA["Pink8"]) + 
  scale_y_continuous_rba(limits = c(100, 300), units = "$b") + 
  source_rba(c("APRA"))

paymentsGraphRDP <- format_rdp(paymentsGraph)

ggrba::ggsave_rba(paymentsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Real-time Gross Settlement", ".png"))
ggrba::ggsave_rba(paymentsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Real-time Gross Settlement", ".svg"))
ggrba::ggsave_rba(paymentsGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Real-time Gross Settlement", ".pdf"))

ggrba::export_excel_rba(gg = paymentsGraphRDP, 
                        filename = paste0(dir, "Figure 4.xlsx"))

## Graph 5 - Short-term runnable liabilities ----------------------------------------------------------

graphNo <- graphNo + 1

outrightsRaw <- readxl::read_excel("not included in supp info.xlsx")

turnoverOutrightActual <- outrightsRaw %>%
  select(date, ags_turnover, semis_turnover) %>% 
  `colnames<-`(c("date", "AGS", "SGS")) %>% 
  pivot_longer(-date, names_to = "issuerType", values_to = "total") %>% 
  mutate(date = as.Date(date), 
         month = rbatools::end_of_month(date)) %>%
  group_by(month, issuerType) %>% 
  summarise(dailyMin = min(total), 
            dailyAvg = mean(total), 
            dailyMax = max(total)) %>% 
  ungroup() %>% 
  mutate(market = "Outright") %>% 
  group_by(month, market) %>% 
  summarise(dailyAvg = sum(dailyAvg), 
            dailyMax = sum(dailyMax)) %>% 
  ungroup()

asxRepo <- fetchASXRepoData(startDate = "2021-01-01")

repoTurnoverASXData <- asxRepo %>% 
  filter(issuerType %in% c("Govt", "Semi-Govt")) %>% 
  mutate(issuerType = case_when(issuerType == "Govt" ~ "AGS", 
                                issuerType == "Semi-Govt" ~ "SGS", 
                                TRUE ~ "Other")) %>% 
  group_by(issuerType, settleDate) %>% 
  summarise(total = sum(leg1CV)) %>% 
  ungroup() %>% 
  mutate(month = rbatools::end_of_month(settleDate)) %>% 
  filter((total > 500e6 & issuerType == "AGS") |
           (total > 100e6 & issuerType == "SGS")) %>% # some small days that are actually just public holidays
  group_by(month, issuerType) %>% 
  summarise(dailyMin = min(total)/1e9, 
            dailyAvg = mean(total)/1e9, 
            dailyMax = max(total)/1e9) %>%
  ungroup() %>% 
  mutate(market = "Repo") %>% 
  group_by(month, market) %>% 
  summarise(dailyAvg = sum(dailyAvg), 
            dailyMax = sum(dailyMax)) %>% 
  ungroup()

finalASXTurnover <- bind_rows(repoTurnoverASXData, turnoverOutrightActual) %>% 
  group_by(market) %>% 
  summarise(`Average daily turnover\n in HQLA securities` = mean(dailyAvg), 
            `Maximum daily turnover\n in HQLA securities` = max(dailyMax)) %>% 
  ungroup() %>% 
  mutate(total = 0)

graphDataTurnover <- finalASXTurnover %>% 
  pivot_longer(-market) %>% 
  drop_na() %>% 
  filter(value != 0) %>% 
  mutate(value = case_when(grepl("Outright", market) ~ round(value, 0), 
                           TRUE ~ round(value))
  )

otherBars <- graphDataTurnover %>% 
  add_row(name = "Excess reserves", 
          market = " ",
          value = ESBalDaily$ESAbalanceSurplus[ESBalDaily$Date == max(ESBalDaily$Date)]/1e3) %>% 
  add_row(
    name = "... of which runnable overnight", 
    market = "  ",
    value = overnightNCOsAndFunding %>%
      drop_na() %>% 
      filter(date == "2024-06-30") %>% 
      pull(overnightNCOs) %>% 
      unique()) %>% 
  add_row(name = "Net cash outflows...", 
          market = "   ",
          value = 703.60353) %>% 
  filter(!name %in% graphDataTurnover$name)

NCOsPanel1 <- ggrba() + 
  geom_col(data = otherBars, 
           mapping = aes(x = reorder(name, -value), y = value, fill = market)) +  
  scale_y_continuous_rba(units = "$b", limits = c(0, 800), 
                         breaks = seq(0, 800, 100)) +  
  ggrba::rotate_x_labels_rba(angle = 90) + 
  theme(axis.text.x=element_text(hjust=0.95)) + 
  scale_fill_manual(values = as.vector(c(RBA["Purple2"], 
                                         RBA["Violet1"],
                                         RBA["DarkGreen3"]))) 

NCOsPanel2 <- ggrba() + 
  geom_col(data = graphDataTurnover, 
           mapping = aes(x = reorder(name, -value), y = value, fill = market)) +  
  scale_y_continuous_rba(units = "$b", limits = c(0, 800), 
                         breaks = seq(0, 800, 100)) + 
  ggrba::rotate_x_labels_rba(angle = 90) + 
  scale_fill_manual(values = as.vector(rbatools::rba_palettes$classic[4:5])) + 
  legend_rba(c(0.8, 0.93)) + 
  theme(axis.text.x=element_text(hjust=0.95)) 

compareNCOsToHQLAMarkets <- NCOsPanel1 + 
  NCOsPanel2 + 
  multipanel_title_rba("blha", "sdf")

compareNCOsToHQLAMarketsRDP <- format_rdp(compareNCOsToHQLAMarkets)

ggrba::ggsave_rba(compareNCOsToHQLAMarketsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Short-term Liabilities and HQLA Market Turnover", ".png"))
ggrba::ggsave_rba(compareNCOsToHQLAMarketsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Short-term Liabilities and HQLA Market Turnover", ".svg"))
ggrba::ggsave_rba(compareNCOsToHQLAMarketsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Short-term Liabilities and HQLA Market Turnover", ".pdf"))

write.csv(bind_rows(otherBars, graphDataTurnover), 
          file = paste0(dir, "Figure 5.csv"))

## Graph 6 - Overnight funding ---------------------------------------------

graphNo <- graphNo + 1

overnightFundingGraph <- overnightNCOsAndFunding %>%
  select(date, overnightFundingShare, overnightNCOsShare) %>% 
  `colnames<-`(c("date", "Unweighted", "LCR-weighted")) %>% 
  filter(date != max(date)) %>% 
  ggrba() + 
  geom_line(aes(x = date, y = Unweighted, colour = "Unweighted")) + 
  geom_line_rhs_rba(aes(x = date, y = `LCR-weighted`, colour = "LCR-weighted")) + 
  scale_y_continuous_rba(units = "%", limits = c(40, 60), 
                         sec.axis = rhs_axis_rba("%", limits = c(10, 14))) +
  scale_colour_manual(values = as.vector(c(RBA["Aqua1"], 
                                           RBA["Olive1"]))) + 
  source_rba(c("APRA", "RBA")) + 
  autolabel_var_rba() + 
  labs(
    title = "LCR Banks' Overnight Funding",
    subtitle = "Share of total balance sheet size"
  )  + 
  annotate("text", 
           x = as.Date("2024-01-01"), 
           y = 47.0025188916877,  label = "LCR-weighted", 
           colour = "#548476ff",  
           size = 7.029,  
           hjust = "centre", vjust = "centre") + 
  annotate("text", 
           x = as.Date("2024-01-01"), 
           y = 45.80025188916877,  label = "(RHS)", 
           colour = "#548476ff",  
           size = 0.9*7.029,  
           hjust = "centre", vjust = "centre") + 
  annotate("text", 
           x = as.Date("2020-06-01"), 
           y = 42.6196473551637,  label = "Unweighted", 
           colour = RBA["Olive1"],  
           size = 7.029,  
           hjust = "centre", vjust = "centre") + 
  annotate("text", 
           x = as.Date("2020-06-01"), 
           y = 41.4196473551637,  label = "(LHS)", 
           colour = RBA["Olive1"],  
           size = 0.9*7.029,  
           hjust = "centre", vjust = "centre")

overnightFundingGraphRDP <- format_rdp(overnightFundingGraph)

ggrba::ggsave_rba(overnightFundingGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "LCR Banks' Overnight Funding", ".png"))
ggrba::ggsave_rba(overnightFundingGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "LCR Banks' Overnight Funding", ".svg"))
ggrba::ggsave_rba(overnightFundingGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "LCR Banks' Overnight Funding", ".pdf"))

ggrba::export_excel_rba(gg = overnightFundingGraphRDP, 
                        filename = paste0(dir, "Figure 6.xlsx"))

## Graph 7 - Relative value demand ----------------------------------------------------------------

graphNo <- graphNo + 1

datez <- tibble(Date = seq(as.Date("2020-01-01"), max(relativeValue$Date), by = 1))

minMaxRelativeValue <- relativeValue %>% 
  group_by(Date) %>%
  mutate(no = n()) %>% 
  filter(no >= 6) %>% 
  group_by(Date) %>% 
  summarise(min = min(AGS)*100, 
            max = max(AGS)*100, 
            AGS = mean(AGS)*100)

relativeValueGraphData <- datez %>% 
  left_join(minMaxRelativeValue) %>% 
  fill(min, max, AGS)

AGS <- ggrba() + 
  geom_ribbon(data = relativeValueGraphData,
              aes(x = Date, ymin = min, ymax = max), 
              fill = RBA["Grey2"]) + 
  geom_line(data = relativeValueGraphData,
            aes(x = Date, y = AGS, colour = "Average"), 
            colour = RBA["Maroon8"]) + 
  scale_y_continuous_rba(units = "bps", limits = c(-80, 80), 
                         breaks = seq(-80, 80, 20)) + 
  source_rba(c("Bloomberg", "RBA")) + 
  autolabel_var_rba() + 
  labs(
    title = "Hedged AGS Returns",
    subtitle = "Average of 1 - 10 year AGS; Relative to the ES Rate"
  ) + 
  geom_hline(aes(yintercept = 0)) + 
  footnote_rba("Range of 1, 2, 3, 5, 7 and 10 year AGS")

hedgedAGSReturn <- ggrba::autolabel_rba(AGS)

hedgedAGSReturnRDP <- format_rdp(hedgedAGSReturn)

ggrba::ggsave_rba(hedgedAGSReturnRDP, file = paste0(dir, "Figure ", graphNo, " - ", "AGS Return", ".png"))
ggrba::ggsave_rba(hedgedAGSReturnRDP, file = paste0(dir, "Figure ", graphNo, " - ", "AGS Return", ".svg"))
ggrba::ggsave_rba(hedgedAGSReturnRDP, file = paste0(dir, "Figure ", graphNo, " - ", "AGS Return", ".pdf"))

write.csv(relativeValueGraphData, file = paste0(dir, "Figure 7.csv"))

## Graph 8 - Trades away from the cash rate  ----------------------------------------------------------------

graphNo <- graphNo + 1

transactionsData <- mpi::fetchCashMarketTransactions(startDate = "2019-01-01",
                                                     endDate = Sys.Date(),
                                                     timeStamps = FALSE,
                                                     database = "IBOC_Prod",
                                                     server = NULL) %>%
  dplyr::mutate(payerName = trimws(.data$payerName),
                receiverName = trimws(.data$receiverName))

tradesNotAtTheCashRate <- transactionsData %>%
  dplyr::filter(.data$dealDate >= as.Date("2019-01-01")) %>%
  dplyr::mutate(awayMid = dplyr::case_when(.data$rate != .data$publishedCashRate ~ 1,
                                           TRUE ~ 0),
                awayMidVolume = dplyr::case_when(.data$rate != .data$publishedCashRate ~ .data$amount,
                                                 TRUE ~ 0),
                bankLevyBanks = dplyr::case_when(.data$payerName %in% c("AUSTRALIA AND NEW ZEALAND BANKING GROUP LIMITED",
                                                                        "COMMONWEALTH BANK OF AUSTRALIA",
                                                                        "NATIONAL AUSTRALIA BANK LIMITED",
                                                                        "WESTPAC BANKING CORPORATION",
                                                                        "MACQUARIE BANK LIMITED") ~ "Bank Levy Banks",
                                                 TRUE ~ "Other")) %>%
  dplyr::mutate(month = rbatools::start_of_month(.data$dealDate)) %>%
  dplyr::group_by(.data$month, .data$bankLevyBanks) %>%
  dplyr::summarise(misses = sum(.data$awayMid),
                   missesVolume = sum(.data$awayMidVolume),
                   totalTrans = dplyr::n(),
                   totalTransVolume = sum(.data$amount)) %>%
  dplyr::group_by(.data$month) %>%
  dplyr::mutate(missesPerc = .data$misses/sum(.data$totalTrans)*100,
                missesPercVolume = .data$missesVolume/sum(.data$totalTransVolume)*100) %>%
  dplyr::mutate(bankLevyBanks = factor(.data$bankLevyBanks, levels = c("Other",
                                                                       "Bank Levy Banks")))

# perc trades above the cash rate (no split)

tradesNotAtCashRateNoSplit <- tradesNotAtTheCashRate %>%
  dplyr::group_by(.data$month) %>%
  dplyr::summarise(misses = sum(.data$misses),
                   totalTrans = sum(.data$totalTrans)) %>%
  dplyr::mutate(missesPerc = .data$misses/.data$totalTrans*100, 
                month = month + 16) %>%
  dplyr::ungroup() %>%
  select(month, missesPerc) %>% 
  ggrba::ggrba(ggplot2::aes(x = month, y = missesPerc)) +
  ggplot2::geom_bar(stat = "identity", position = "stack", 
                    fill = RBA["Blue3"]) +
  ggplot2::labs(x = "Date",
                y = "%",
                title = "Trades Away from the Cash Rate",
                subtitle = "Proportion; by volume; since the pandemic"
  ) +
  ggrba::legend_rba("topleft") +
  ggplot2::scale_fill_manual(values = as.vector(c(RBA["Blue3"]))) +
  ggplot2::guides(fill = "none") +
  ggrba::scale_y_continuous_rba(units = "%",
                                limits = c(0, 60)) +
  ggrba::source_rba("RBA")

tradesNotAtCashRateNoSplitRDP <- format_rdp(tradesNotAtCashRateNoSplit)

ggrba::ggsave_rba(tradesNotAtCashRateNoSplitRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Trades Away from the Cash Rate", ".png"))
ggrba::ggsave_rba(tradesNotAtCashRateNoSplitRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Trades Away from the Cash Rate", ".svg"))
ggrba::ggsave_rba(tradesNotAtCashRateNoSplitRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Trades Away from the Cash Rate", ".pdf"))

ggrba::export_excel_rba(gg = tradesNotAtCashRateNoSplitRDP, 
                        filename = paste0(dir, "Figure 8.xlsx"))

## Graph 9 - ASX vs. SOFIA ----------------------------------------------------------------

graphNo <- graphNo + 1

SOFIAGraph <- finalData %>% 
  filter(tradeDate >= as.Date("2020-11-01"), 
         tradeDate <= as.Date("2024-08-31")) %>% 
  select(tradeDate, APRASOFIASpread, SOFIAspreadMedian) %>% 
  `colnames<-`(c("tradeDate", "Calculated using APRA data", "SOFIA*")) %>% 
  pivot_longer(-tradeDate) %>% 
  drop_na() %>% 
  ggrba() + 
  geom_line(aes(x = tradeDate, y = value, colour = name, 
                .autolabel = name)) + 
  scale_y_continuous_rba(units = "bps") + 
  geom_hline(aes(yintercept = 0)) + 
  autolabel_var_rba() + 
  labs(
    title = "Overnight GC1 Repo Rate",
    subtitle = "Spread to ES Rate; by data source"
  ) + 
  footnote_rba("Volume-weighted median") + 
  source_rba(c("APRA", "ASX", "RBA")) + 
  scale_colour_manual(values = as.vector(c(RBA["Blue5"], 
                                           RBA["Orange5"])))

SOFIAGraphLabelled <- ggrba::autolabel_rba(SOFIAGraph)

SOFIAGraphRDP <- format_rdp(SOFIAGraphLabelled)

ggrba::ggsave_rba(SOFIAGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Overnight GC Repo Rate", ".png"))
ggrba::ggsave_rba(SOFIAGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Overnight GC Repo Rate", ".svg"))
ggrba::ggsave_rba(SOFIAGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Overnight GC Repo Rate", ".pdf"))

cor(finalData$APRASOFIAVWAPSpread[finalData$tradeDate >= as.Date("2022-01-01")], 
    finalData$SOFIAspreadMedian[finalData$tradeDate >= as.Date("2022-01-01")])

write.csv(finalData %>% 
            filter(tradeDate >= as.Date("2020-11-01"), 
                   tradeDate <= as.Date("2024-08-31")) %>% 
            select(tradeDate, APRASOFIASpread, SOFIAspreadMedian) %>% 
            `colnames<-`(c("tradeDate", "Calculated using APRA data", "SOFIA*")), 
          file = paste0(dir, "Figure 9.csv"))

## Graph 10 - Specials ---------------------------------------------------------------

graphNo <- graphNo + 1

repoMeasureForGraphing <- repoRatesByEntityCurrent %>%
  mutate(marketType = "Private market") %>% 
  left_join(APRASOFIACalc, by = "tradeDate") %>% 
  mutate(specialsSpread = (APRASOFIA - rate)*100) %>% 
  select(tradeDate, rate, APRASOFIA, specialsSpread, collMV, everything()) %>% 
  mutate(specials = case_when(specialsSpread > 10 ~ 1, 
                              TRUE ~ 0)) %>% 
  bind_rows(RBASLToBind %>% 
              mutate(specials = 1, 
                     marketType = "RBA securities lending")) %>% 
  dplyr::group_by(.data$tradeDate, marketType, specials) %>%
  summarise(marketValue = sum(collMV)) %>% 
  group_by(tradeDate) %>% 
  mutate(share = marketValue/sum(marketValue)*100) %>% 
  ungroup() %>% 
  filter(specials == 1, 
         tradeDate <= as.Date("2024-08-31"), 
         tradeDate >= as.Date("2020-10-01")) %>% 
  mutate(tradeDate = tradeDate - 15)

specialsMeasure <- ggrba() +
  geom_col(data = repoMeasureForGraphing %>% 
             mutate(marketType = factor(marketType, levels = c("RBA securities lending", 
                                                               "Private market"))),
           mapping = aes(x = tradeDate, y = share, fill = marketType)) +
  labs(title = "title", 
       subtitle = "subtitle") + 
  scale_fill_manual(values = as.vector(c(RBA["Olive5"], RBA["Purple6"]))) + 
  scale_y_continuous_rba(units = "%") + 
  scale_x_date_rba() +
  source_rba(c("APRA", "RBA")) + 
  annotate("text", 
           x = as.Date("2021-03-01"), y = 50, label = "RBA securities\n lending",
           size = 7.029, 
           colour = RBA["Olive5"]) + 
  annotate("text", 
           x = as.Date("2020-07-01"), y = 10, label = "Private\n market",
           size = 7.029, 
           colour = RBA["Purple6"]) 

specialsMeasureRDP <- format_rdp(specialsMeasure)

ggrba::ggsave_rba(specialsMeasureRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Proportion of Bonds Trading Special", ".png"))
ggrba::ggsave_rba(specialsMeasureRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Proportion of Bonds Trading Special", ".svg"))
ggrba::ggsave_rba(specialsMeasureRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Proportion of Bonds Trading Special", ".pdf"))

write.csv(repoMeasureForGraphing %>% 
            select(tradeDate, marketType, share) %>% 
            pivot_wider(names_from = marketType, values_from = share),
          file = paste0(dir, "Figure 10.csv"))

## Graph 11 - Asset swaps --------------------------------------------------

graphNo <- graphNo + 1

minMaxAssetSwaps <- relativeValue %>% 
  group_by(Date) %>%
  mutate(no = n()) %>% 
  filter(no >= 6) %>% 
  group_by(Date) %>% 
  summarise(min = min(ASW), 
            max = max(ASW), 
            ASW = mean(ASW))

specialsCorrelationASW <- ggrba() + 
  geom_ribbon(data = minMaxAssetSwaps %>% 
                filter(Date >= as.Date("2020-11-01")),
              aes(x = Date, ymin = min, ymax = max), 
              fill = RBA["Grey2"]) + 
  geom_line(data = minMaxAssetSwaps %>% 
              filter(Date >= as.Date("2020-11-01")),
            aes(x = Date, y = ASW, colour = "Average"), 
            colour = RBA["Maroon8"]) + 
  geom_line_rhs_rba(data = robustnessData, mapping = aes(x = tradeDate, y = -APRAVolumeGCMinus10, 
                                                         colour = "Specials"), 
                    colour = RBA["Olive3"]) + 
  scale_y_continuous_rba(units = "bps", 
                         limits = c(-100, 20), 
                         sec.axis = ggrba::rhs_axis_rba(units = "%", limits = c(-100, 20), 
                                                        labels = c("100", "80", "60", "40", "20", "0", "-20"))) + 
  annotate("text", 
           x = as.Date("2024-01-31"), 
           y = 10, 
           label = "'Specials,'~inverted^{(b)}", 
           parse = TRUE,
           colour = RBA["Olive3"], 
           size = 7.029) + 
  annotate("text", 
           x = as.Date("2024-01-31"), 
           y = 4.5, 
           label = "(RHS)", 
           parse = TRUE,
           colour = RBA["Olive3"], 
           size = 18/20*7.029) + 
  annotate("text", 
           x = as.Date("2021-05-31"), 
           y = 13, 
           label = paste0("'Asset swap", "'~spread^{(a)}"),
           parse = TRUE,
           colour = RBA["Maroon8"], 
           size = 7.029) + 
  annotate("text", 
           x = as.Date("2021-05-31"), 
           y = 6.5, 
           label = "(LHS)", 
           parse = TRUE,
           colour = RBA["Maroon8"],
           size = 18/20*7.029) + 
  source_rba(c("APRA", "Bloomberg", "RBA")) + 
  footnote_rba("Range of 1, 2, 3, 5, 7 and 10 year AGS swap spreads") + 
  footnote_rba("Proportion of repo volume trading special; including RBA securities lending facility") +
  labs(title = "Specials and Asset Swap Spreads", 
       subtitle = "Hey")

specialsCorrelationASWRDP <- format_rdp(specialsCorrelationASW)

ggrba::ggsave_rba(specialsCorrelationASWRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Specials and Asset Swap Spreads", ".png"))
ggrba::ggsave_rba(specialsCorrelationASWRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Specials and Asset Swap Spreads", ".svg"))
ggrba::ggsave_rba(specialsCorrelationASWRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Specials and Asset Swap Spreads", ".pdf"))

write.csv(robustnessData %>%
            select(tradeDate, APRAVolumeGCMinus10), 
          file = paste0(dir, "Figure 11.csv"))

## Graph 12 - Reserve demand ---------------------------------------------------------------

graphNo <- graphNo + 1

pathData <- IVData %>% 
  select(tradeDate, 
         ESBal, 
         APRASOFIASpread) %>% 
  drop_na() %>% 
  left_join(spreadToESPredictedIV, by = "tradeDate") %>%
  `colnames<-`(c("tradeDate", 
                 "ESBal", 
                 "Actual",
                 "Predicted - only reserves",
                 "Predicted - full model")) %>% 
  pivot_longer(-c(tradeDate, ESBal)) 

panel1 <- pathData %>% 
  filter(name == "Actual" | grepl("only reserves", name)) %>%
  mutate(name = gsub(" - only reserves", "", name)) %>% 
  drop_na() %>% 
  ggrba(aes(x = ESBal, y = value, label = tradeDate, col = name)) + 
  geom_point(size = 1.2) + 
  geom_path() + 
  scale_color_manual(values = as.vector(c(RBA["Blue5"],
                                          RBA["Green3"]))) +  
  axis_titles_rba() + 
  labs(title = "Reserves only", 
       y = "Overnight repo rate spread to ES rate \U2012 bps", 
       x = "Excess reserves \U2012 $b") + 
  scale_x_continuous_rba(units = " ", limits = c(0, 500)) + 
  scale_y_continuous_rba(units = "20", limits = c(-5, 20))

panel2 <- pathData %>% 
  filter(name == "Actual" | grepl("full model", name)) %>%
  mutate(name = gsub(" - full model", "", name)) %>% 
  drop_na() %>% 
  ggrba(aes(x = ESBal, y = value, label = tradeDate, col = name, 
            .autolabel = name)) + 
  geom_point(size = 1.2) + 
  geom_path() + 
  scale_color_manual(values = as.vector(c(RBA["Blue5"],
                                          RBA["Green3"]))) + 
  axis_titles_rba(ytitle = FALSE) + 
  labs(title = "Full model", 
       x = "Excess reserves \U2012 $b", 
       y = "Overnight repo rate spread to ES rate \U2012 bps") + 
  scale_x_continuous_rba(units = " ", limits = c(0, 500)) + 
  scale_y_continuous_rba(units = "20", limits = c(-5, 20)) + 
  ggplot2::theme(
    axis.title.y.right = ggplot2::element_text(
      size = 20, angle = 90, vjust = -1,
      margin = grid::unit(c(0, 1, 0, 0), "mm")))

twoPanelGraph <- panel1 + 
  panel2 + 
  source_rba(c("APRA", "RBA")) + 
  patchwork::plot_layout(axis_titles = "collect")

twoPanelGraphLabelled <- ggrba::autolabel_rba(twoPanelGraph)

twoPanelGraphRDP <- format_rdp(twoPanelGraphLabelled)

ggrba::ggsave_rba(twoPanelGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Overnight Repo Rate", ".png"))
ggrba::ggsave_rba(twoPanelGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ",  "Overnight Repo Rate", ".svg"))
ggrba::ggsave_rba(twoPanelGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ",  "Overnight Repo Rate", ".pdf"))

ggrba::export_excel_rba(gg = twoPanelGraphRDP, 
                        filename = paste0(dir, "Figure 12.xlsx"))

## Graph 13 - Collateral dynamics ------------------------------------------

graphNo <- graphNo + 1

collateralDynIV <- IVData %>% 
  select(tradeDate, RBASLOutstanding, APRAVolumeGCMinus10) %>% 
  mutate(predict = APRAVolumeGCMinus10*dIVCD, 
         lowerCI = predict - 1.98*IVRegForPred$se[4], 
         upperCI = predict + 1.98*IVRegForPred$se[4]) 

collDynGraphData <- collateralDynIV %>% 
  select(tradeDate, predict) %>% 
  `colnames<-`(c("tradeDate", "Estimated")) %>% 
  pivot_longer(-tradeDate)

collDynIVGraph <- ggrba() + 
  labs(title = "Opportunity Cost of Collateral",
       subtitle = "Effect of collateral shortages on the overnight GC repo rate") +
  scale_y_continuous_rba(units = "bps", limits = c(-5, 1)) +
  scale_x_date_rba() + 
  geom_line(data = collDynGraphData, 
            mapping = aes(x = tradeDate, 
                          y = value), 
            colour = RBA["Red3"]) + 
  geom_hline(yintercept = 0) + 
  source_rba(c("ASX", "RBA"))  

collDynIVGraphRDP <- ggrba::format_rdp(collDynIVGraph)

ggrba::ggsave_rba(collDynIVGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "The Cost of Collateral", ".png"))
ggrba::ggsave_rba(collDynIVGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "The Cost of Collateral", ".svg"))
ggrba::ggsave_rba(collDynIVGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "The Cost of Collateral", ".pdf"))

write.csv(collateralDynIV %>% 
            select(tradeDate, predict) %>% 
            `colnames<-`(c("tradeDate", "Estimated")),
          file = paste0(dir, "Figure 13.csv"))

## Graph 14 - Reserve demand with error bands----------------------------------------------------------------

finalDep <- IVData$depositMeasure[IVData$tradeDate == max(IVData$tradeDate)]

finalVolBelowGC10 <- IVData$APRAVolumeGCMinus10[IVData$tradeDate == max(IVData$tradeDate)]

errorsCD <- list()

for(i in 1:500){
  
  xCD <- c(rep(1, 1), 
           c(log(i)), 
           rep(log(finalDep), 1), 
           rep(finalVolBelowGC10, 1)
  )
  
  dim(xCD) <- c(1, 4)
  
  errorsCD[[i]] <- tibble(reserves = i, 
                          SE = xCD %*% vcovHAC(IVRegForPred) %*% t(xCD) %>% 
                            as.numeric(), 
                          model = "Base")
  
}

errorBandsData <- bind_rows(errorsCD) %>% 
  mutate(Deposits = finalDep,
         finalVolBelowGC10 = finalVolBelowGC10,
         fitted = aIVCD + 
           bIVCD*log(reserves) + 
           cIVCD*log(Deposits) + 
           dIVCD*finalVolBelowGC10, 
         CIMin = fitted - 1.645*sqrt(SE), 
         CIMax = fitted + 1.645*sqrt(SE))  

# Estimates @ OMO price 

estimates <- errorBandsData %>% 
  select(reserves, fitted, CIMin, CIMax) %>% 
  pivot_longer(-c(reserves)) %>% 
  group_by(name) %>% 
  filter(value <= 15) %>% 
  filter(value == max(value)) %>%
  ungroup() %>% 
  select(-value) %>% 
  pivot_wider(names_from = name, values_from = reserves)

graphNo <- graphNo + 1

demandWithErrorBands <- errorBandsData %>%
  mutate(CIMax = case_when(CIMax > 40 ~ 40, 
                           TRUE ~ CIMax)) %>% 
  ggrba() + 
  labs(title = "Reserves Demand Curve",
       subtitle = "As at end-August 2024",
       x = "Excess reserves \U2012 $b", 
       y = "Overnight repo rate spread to ES rate \U2012 bps") + 
  scale_y_continuous_rba(units = "40", 
                         limits = c(0, 40), 
                         sec.axis = rhs_axis_rba("", scalar = 1, 
                                                 labels = rep("", 5))) + 
  scale_x_continuous_rba(units = "500") +
  axis_titles_rba() + 
  geom_ribbon(aes(x = reserves,
                  ymin = CIMin, 
                  ymax = CIMax, 
                  fill = model), 
              fill = as.vector(RBA["Grey2"])) +
  geom_line(aes(x = reserves, 
                y = fitted, 
                colour = model)) + 
  scale_colour_manual(values = as.vector(c(RBA["Grey10"]))) + 
  geom_hline(yintercept = 15, colour = "red", lwd = 1) +
  annotate("text", 
           x = 400,
           y = 17,
           label = "Current OMO price",
           size = 7.029,
           colour = "red") + 
  source_rba("RBA") + 
  footnote_rba("Shaded regions represent 90% confidence intervals")

demandWithErrorBandsRDP <- format_rdp(demandWithErrorBands)

ggrba::ggsave_rba(demandWithErrorBandsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".png"))
ggrba::ggsave_rba(demandWithErrorBandsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".svg"))
ggrba::ggsave_rba(demandWithErrorBandsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".pdf"))

ggrba::export_excel_rba(gg = demandWithErrorBandsRDP, 
                        filename = paste0(dir, "Figure 14.xlsx"))

## Graph 15 - then and now --------------------------------------------------

graphNo <- graphNo + 1

errorsOverTime <- list()

for(j in sort(unique(IVData$tradeDate))){
  
  depMeasure <- IVData %>% 
    filter(tradeDate == j) %>% 
    pull(depositMeasure)
  
  collMeasure <- IVData %>% 
    filter(tradeDate == j) %>% 
    pull(APRAVolumeGCMinus10)
  
  for(i in 1:500){
    
    xCD <- c(rep(1, 1), 
             c(log(i)), 
             rep(log(depMeasure), 1), 
             rep(collMeasure, 1)
    )
    
    dim(xCD) <- c(1, 4)
    
    errorsOverTime[[paste0(i, j)]] <- tibble(reserves = i, 
                                             SE = xCD %*% vcovHAC(IVRegForPred) %*% t(xCD) %>% 
                                               as.numeric(),
                                             tradeDate = j,
                                             dep = depMeasure, 
                                             coll = collMeasure, 
                                             model = "Collateral Dyn")
    
  }
  
}

demandThenAndNow <- bind_rows(errorsOverTime) %>% 
  mutate(tradeDate = lubridate::as_date(tradeDate)) %>% 
  left_join(IVData) %>% 
  mutate(fitted = aIVCD + 
           bIVCD*log(reserves) + 
           cIVCD*log(dep) + 
           dIVCD*coll, 
         CIMin = fitted - 1.645*sqrt(SE), 
         CIMax = fitted + 1.645*sqrt(SE)) %>%
  select(tradeDate, reserves, fitted, CIMin, CIMax) %>% 
  arrange(tradeDate, reserves) %>% 
  filter(tradeDate == max(tradeDate) | tradeDate == min(tradeDate)) %>% 
  mutate(CIMax = case_when(CIMax > 40 ~ 40, 
                           TRUE ~ CIMax)) %>% 
  ggrba() + 
  labs(
    title = "Reserves Demand Curve",
    subtitle = "November 2020 vs. August 2024",
    x = "Excess reserves \U2012 $b", 
    y = "Overnight repo rate spread to ES rate \U2012 bps") + 
  scale_y_continuous_rba(units = "40", 
                         limits = c(-10, 40), 
                         sec.axis = rhs_axis_rba("", scalar = 1, 
                                                 labels = rep("", 6))) + 
  scale_x_continuous_rba(units = "500") +
  axis_titles_rba() + 
  geom_ribbon(aes(x = reserves,
                  ymin = CIMin,
                  ymax = CIMax,
                  fill = as.character(tradeDate))) +
  geom_line(aes(x = reserves, 
                y = fitted, 
                colour = as.character(tradeDate), 
                .autolabel = format(tradeDate, "%b %Y"))) + 
  scale_colour_manual(values = as.vector(c(RBA["Blue10"], 
                                           RBA["Grey10"]))) + 
  scale_fill_manual(values = as.vector(c(RBA["Blue2"], 
                                         RBA["Grey2"]))) + 
  geom_hline(yintercept = 15, colour = "red", lwd = 1) +
  annotate("text", 
           x = 400,
           y = 17,
           label = "Current OMO price",
           size = 7.029,
           colour = "red") + 
  footnote_rba("Shaded regions represent 90% confidence intervals") + 
  autolabel_var_rba()

demandThenAndNow <- ggrba::autolabel_rba(demandThenAndNow)

demandThenAndNowRDP <- format_rdp(demandThenAndNow)

ggrba::ggsave_rba(demandThenAndNowRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Shift in Banks' Reserve Demand Curve", ".png"))
ggrba::ggsave_rba(demandThenAndNowRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Shift in Banks' Reserve Demand Curve", ".svg"))
ggrba::ggsave_rba(demandThenAndNowRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Shift in Banks' Reserve Demand Curve", ".pdf"))

ggrba::export_excel_rba(gg = demandThenAndNowRDP, 
                        filename = paste0(dir, "Figure 15.xlsx"))

## Graph 16 - Projection of ES balances ------------------------------------

graphNo <- graphNo + 1

BSForecast <- forecastBS()

minESAReq <- fetchOpenRepo() %>% 
  pull(MinESArequirement) %>% 
  sum(na.rm = TRUE)

BSForecastToJoin <- BSForecast$weeklyBS %>% 
  filter(type == "forecast") %>% 
  mutate(type = "Forecast") %>% 
  select(date, type, totalESB, minReqESBForecast) %>% 
  ungroup() %>% 
  mutate(surplusESB = as.numeric(totalESB - minReqESBForecast))

ESBalToJoin <- ESBalDaily %>%
  mutate(type = "Actual") %>% 
  select(Date, type, ESAbalanceEOD, ESAbalanceSurplus) %>% 
  `colnames<-`(c("date", "type", "totalESB", "surplusESB"))

DateMin <- BSForecastToJoin %>% 
  filter(surplusESB <= estimates$CIMin*1e3) %>% 
  filter(date == min(date)) %>% 
  pull(date)

DateMax <- BSForecastToJoin %>% 
  filter(surplusESB <= estimates$CIMax*1e3) %>% 
  filter(date == min(date)) %>% 
  pull(date)

DataForESBalForecast <- ESBalToJoin %>% 
  bind_rows(BSForecastToJoin) %>% 
  filter(date <= as.Date("2028-12-31"))

ESBalGraph <- ggrba() + 
  geom_rect(aes(xmin = DateMax, 
                xmax = as.Date("2028-12-31"), 
                ymin = estimates$CIMin, 
                ymax = estimates$CIMax), 
            fill = arphit::RBA["Grey2"]) + 
  geom_line(data = DataForESBalForecast %>% 
              filter(type == "Actual"), 
            mapping = aes(x = date, y = surplusESB/1e3, colour = "Actual"), 
            lty = 1, 
            colour = arphit::RBA["Default1"]) + 
  geom_line(data = DataForESBalForecast %>% 
              filter(type == "Forecast"), 
            mapping = aes(x = date, y = surplusESB/1e3, colour = "Forecast"), 
            lty = 2,
            colour = arphit::RBA["Default2"]) + 
  annotate("text", 
           label = "Actual", 
           x = as.Date("2020-10-01"), 
           y = 350, 
           size = 7.029, 
           colour = arphit::RBA["Default1"]) + 
  annotate("text", 
           label = "Projected*", 
           x = as.Date("2027-04-30"), 
           y = 50,
           size = 7.029, 
           colour = arphit::RBA["Default2"]) + 
  annotate("text", 
           label = "Range of\nreserve demand", 
           x = as.Date("2023-08-01"), 
           y = 150,
           size = 7.029,
           colour = arphit::RBA["Grey3"]) + 
  scale_y_continuous_rba(units = "$b", limits = c(0, 500)) + 
  labs(title = "Reserve Supply",
       subtitle = "Actual, projected and reserve demand") +
  source_rba(c("RBA")) +
  footnote_rba("Assumes RBA bond holdings are held to maturity and no net change in RBA liquidity operations.")

ESBalGraphRDP <- ggrba::format_rdp(ESBalGraph)

ggrba::ggsave_rba(ESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Excess Reserves", ".png"))
ggrba::ggsave_rba(ESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Excess Reserves", ".svg"))
ggrba::ggsave_rba(ESBalGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Excess Reserves", ".pdf"))

write.csv(DataForESBalForecast %>% 
            select(date, surplusESB, type) %>% 
            mutate(surplusESB = surplusESB/1e3) %>% 
            pivot_wider(names_from = type, values_from = surplusESB), 
          file = paste0(dir, "Figure 16.csv"))

## Graph 17 - OMO Outstanding ----------------------------------------------------------------

graphNo <- graphNo + 1

OMOTotalWeekly <- generateOMOCashDealtChartData(startDate = "2019-01-01") 

omoGraph <- ggrba() + 
  geom_line(data = omoOutstandingVolume, 
            aes(x = asAt, y = OMO),
            colour = RBA["Green8"]
  ) + 
  scale_y_continuous_rba(units = "$b") + 
  source_rba("RBA") +
  labs(title = "OMO Repo Outstanding", 
       subtitle = "Subtitle")

omoGraphRDP <- format_rdp(omoGraph)

ggrba::ggsave_rba(omoGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "OMO Repo Outstanding", ".png"))
ggrba::ggsave_rba(omoGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "OMO Repo Outstanding", ".svg"))
ggrba::ggsave_rba(omoGraphRDP, file = paste0(dir, "Figure ", graphNo, " - ", "OMO Repo Outstanding", ".pdf"))

write.csv(omoOutstandingVolume, 
          file = paste0(dir, "Figure 17.csv"))

## Graph 18 - comparing functional forms  ----------------------------------

graphNo <- graphNo + 1

errorsLog <- list()
errorsInverse <- list()
errorsArcTan <- list()
errorsExp <- list()

for(i in 1:500){
  
  xLog <- c(rep(1, 1), 
            c(log(i)), 
            rep(log(finalDep), 1), 
            rep(finalVolBelowGC10, 1)
  )
  
  dim(xLog) <- c(1, 4)
  
  xInverse <- c(rep(1, 1), 
                c(1/i), 
                rep(log(finalDep), 1), 
                rep(finalVolBelowGC10, 1)
  )
  
  dim(xInverse) <- c(1, 4)
  
  xArcTan <- c(rep(1, 1), 
               c(atan(i)), 
               rep(log(finalDep), 1), 
               rep(finalVolBelowGC10, 1)
  )
  
  dim(xArcTan) <- c(1, 4)
  
  xExp <- c(rep(1, 1), 
            c(exp(-i/1000)), 
            rep(log(finalDep), 1), 
            rep(finalVolBelowGC10, 1)
  )
  
  dim(xExp) <- c(1, 4)
  
  errorsLog[[i]] <- tibble(reserves = i, 
                           SE = xLog %*% vcovHAC(IVRegForPred) %*% t(xLog) %>% 
                             as.numeric(), 
                           fitted = aIVCD + 
                             bIVCD*log(i) + 
                             cIVCD*log(finalDep) + 
                             dIVCD*finalVolBelowGC10,
                           model = "Log")
  
  errorsInverse[[i]] <- tibble(reserves = i, 
                               SE = xInverse %*% vcovHAC(robustnessFFInverse) %*% t(xInverse) %>% 
                                 as.numeric(),
                               fitted = robustnessFFInverse$coefficients["(Intercept)"] + 
                                 robustnessFFInverse$coefficients["I(1/ESBal)"]*(1/i) + 
                                 robustnessFFInverse$coefficients["log(depositMeasure)"]*log(finalDep) + 
                                 robustnessFFInverse$coefficients["APRAVolumeGCMinus10"]*finalVolBelowGC10,
                               model = "Inverse")
  
  errorsArcTan[[i]] <- tibble(reserves = i, 
                              SE = xArcTan %*% vcovHAC(robustnessFFArcTan) %*% t(xArcTan) %>% 
                                as.numeric(), 
                              fitted = robustnessFFArcTan$coefficients["(Intercept)"] + 
                                robustnessFFArcTan$coefficients["atan(ESBal)"]*(atan(i)) + 
                                robustnessFFArcTan$coefficients["log(TotalDepositsAPRA)"]*log(finalDep) + 
                                robustnessFFArcTan$coefficients["APRAVolumeGCMinus10"]*finalVolBelowGC10,
                              model = "Arctan")
  
}

errorBandsFF <- bind_rows(errorsLog) %>%
  bind_rows(bind_rows(errorsArcTan)) %>%
  mutate(Deposits = finalDep,
         finalVolBelowGC10 = finalVolBelowGC10,
         CIMin = fitted - 1.645*sqrt(SE), 
         CIMax = fitted + 1.645*sqrt(SE))  

# Estimates @ OMO price 

estimates <- errorBandsData %>% 
  select(reserves, fitted, CIMin, CIMax) %>% 
  pivot_longer(-c(reserves)) %>% 
  group_by(name) %>% 
  filter(value <= 15) %>% 
  filter(value == max(value)) %>%
  ungroup() %>% 
  select(-value) %>% 
  pivot_wider(names_from = name, values_from = reserves)


diffFFsINT <- errorBandsFF %>%
  mutate(CIMax = case_when(CIMax > 50 ~ 50, 
                           TRUE ~ CIMax)) %>% 
  filter(fitted <= 50) %>%
  ggrba() + 
  labs(
    title = "Reserves Demand Curve",
    subtitle = "As at end-August 2024",
    x = "Excess reserves \U2012 $b", 
    y = "Overnight repo rate spread to ES rate \U2012 bps") + 
  annotate("text", 
           x = 400,
           y = 17,
           label = "Current OMO price",
           size = 7.029,
           colour = "red") + 
  scale_y_continuous_rba(units = "50", 
                         limits = c(0, 50), 
                         sec.axis = rhs_axis_rba("", scalar = 1, 
                                                 labels = rep("", 6))) +
  scale_x_continuous_rba(units = "500") +
  scale_colour_manual(values = as.vector(c(RBA["Purple1"], 
                                           RBA["Grey10"]))) + 
  scale_fill_manual(values = as.vector(c(RBA["Purple1"], 
                                         RBA["Grey10"]))) + 
  axis_titles_rba() + 
  geom_line(aes(x = reserves, 
                y = fitted, 
                colour = model)) + 
  geom_ribbon(aes(x = reserves,
                  ymin = CIMin,
                  ymax = CIMax,
                  fill = model), 
              alpha = 0.2) +
  annotate("text", 
           x = 110, 
           y = 25, 
           label = "Arctan", 
           colour = RBA["Purple1"], 
           size = 7.029) + 
  annotate("text", 
           x = 380, 
           y = 8, 
           label = "Log", 
           colour = RBA["Grey10"], 
           size = 7.029) + 
  geom_hline(yintercept = 15, colour = "red", lwd = 1) +
  source_rba("RBA") + 
  footnote_rba("Shaded regions represent 90% confidence intervals")

diffFFs <- ggrba::autolabel_rba(diffFFsINT)

diffFFsRDP <- ggrba::format_rdp(diffFFs)

ggrba::ggsave_rba(diffFFsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".png"))
ggrba::ggsave_rba(diffFFsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".svg"))
ggrba::ggsave_rba(diffFFsRDP, file = paste0(dir, "Figure ", graphNo, " - ", "Reserve Demand Curve", ".pdf"))

ggrba::export_excel_rba(gg = diffFFsRDP, 
                        filename = paste0(dir, "Figure 18.xlsx"))
